/*************************************************/
/* Examples Program For "ET-dsPIC30F2010"  Board */
/* Hardware    : ET-dsPIC30F2010 TRAINING KIT V1 */
/* Target MCU  : dsPIC30F2010                    */
/*       	   : X-TAL : 7.3728 MHz              */
/*             : Run 117.9648MHz                 */
/*             : Selec OSC Mode = XT w/PLL 16x   */
/* Compiler    : MPLAB + C30 V1.33  		     */
/* Last Update : 1/October/2005                  */
/* Function    : Example Use I2C Connect 24LC32  */
/*             : Write/Verify 24LC32 ---> UART   */	
/*************************************************/
/* Used ET-MINI I/O RTC:DS1307 Module Interface  */
/* SDA = RF2(I2C-SDA)                            */
/* SCL = RF3(I2C-SCL)                            */
/*************************************************/

#include "p30f2010a2.h"                                		// dsPIC30F2010 MPU Register
#include "i2c.h"											// Used I2C Function Library
#include "uart.h"											// Used UART Config
#include "stdio.h"											// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC30F2010 */
_FOSC(CSW_FSCM_ON & XT_PLL16);								// Enable Clock Switching,Enable Fail-Salf Clock
                                                            // Closk Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_ON & BORV_45 & PWRT_64 & MCLR_EN);			// Enable Brown-Out = 4.5V,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End   Configuration For ET-dsPIC30F2010 */

char uart_buf[40]; 											// "sprintf" Buffer

/* pototype  section */
void init_uart(void);										// Initial UART = 9600,N,8,1
void init_i2c(void);										// Initial I2C For Interface DS1307
unsigned char ReadDS1307(unsigned char Read_Addr);			// Read RTC:DS1307 Time
void WriteDS1307(unsigned char Write_Addr,					// Write RTC:DS1307 Time
                unsigned char Set_Time);		


int main(void)
{  
  unsigned char Hour,Minute,Second,Time; 					// RTC:DS1307 Buffer Data
  unsigned char Last_Second;  
  init_uart();												// Initial UART = 9600,N,8,1
  init_i2c();												// Initial I2C Function
   
  sprintf(uart_buf,"ET-dsPIC30F2010 TRAINING KIT V1\n\r"); 	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete
  
  sprintf(uart_buf,"Demo Interface I2C RTC : DS1307\n\n\r");// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  

  //*************************************//
  // Start Setup RTC : DS1307 = 00:00:00 //
  //*************************************//
  WriteDS1307(0x02,0x00);									// Setup Hour = 00	
  WriteDS1307(0x01,0x00);									// Setup Minute = 00
  WriteDS1307(0x00,0x00);									// Setup Second = 00
  Last_Second = 59;											// Default Second

  //********************************//
  // Start Read RTC Clock & Display //
  //********************************//
  while(1)
  { 
    do 														// Repeat Get Second until Second Change
	{
	  Hour   = ReadDS1307(0x02);							// Read Hour (00-23)
	  Minute = ReadDS1307(0x01);							// Read Minute (00-59)
      Second = ReadDS1307(0x00);							// Read Second (00-59)
    }
    while(Last_Second == Second);							// Repeat if Second Not Change
	
    Last_Second = Second;									// Update Current Second
          
    //************************************//
    // Display Clock = Hour:Minute:Second //
    //************************************//
    sprintf(uart_buf,"\rDS1307 Clock = ");					// Print Message String 
  	putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
  	while(BusyUART1());										// Wait putsUART1 Complete
   
	WriteUART1(Hour>>4&0x03|0x30);			
    while(BusyUART1());										// Wait putsUART1 Complete   
    WriteUART1(Hour&0x0F|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete

	WriteUART1(':');	
    while(BusyUART1());										// Wait putsUART1 Complete
    
	WriteUART1(Minute>>4&0x07|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete    
    WriteUART1(Minute&0x0F|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete

    WriteUART1(':');
    while(BusyUART1());										// Wait putsUART1 Complete	
   
	WriteUART1(Second>>4&0x07|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete    
    WriteUART1(Second&0x0F|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete    
  }
}

/*********************************/
/* Initial UART for dsPIC30F2010 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart()
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  
  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
  			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-dsPIC30F2010 TRAINING KIT V1.0 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-dsPIC30F2010 UART Baudrate = 9600 BPS
}

/*************************************/
/* Initial dsPIC30F2010 I2C Function */
/* For Interface DS1307 (I2C RTC)    */
/*************************************/
void init_i2c(void)
{  
  CloseI2C();												// Close I2C Before New Config  

  // Open I2C Function For Interface Ds1307
  OpenI2C(I2C_ON &											// Enable I2C Function
		  I2C_IDLE_STOP &									// Disable I2C in IDLE Mode
		  I2C_CLK_HLD &										// I2C Clock Hold
		  I2C_IPMI_DIS &     								// Disable I2C IPMI Mode Control
		  I2C_7BIT_ADD &									// I2C Device Address = 7 Bit
		  I2C_SLW_DIS &										// Disable I2C Slew Rate Control
		  I2C_SM_DIS &										// Disable I2C SMBUS Mode
		  I2C_GCALL_DIS &									// Disable I2C General Call(Slave) 
          I2C_STR_DIS &										// Disable SCL Clock Stretch
          I2C_ACK &											// ACK Cycle = ACK
	      I2C_ACK_DIS &										// Disable I2C Acknowledge
		  I2C_RCV_DIS &										// Disable I2C Receive
		  I2C_STOP_DIS &									// Disable I2C Stop		
		  I2C_RESTART_DIS &									// Disable I2C Restart
		  I2C_START_DIS,									// Disable I2C Start		

          // ET-dsPIC30F2010 TRAINING KIT V1.0 Hardware Board
		  // XTAL = 7.3728MHz
  		  // Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  		  // Fcy(I2C) = Fosc / 4 
  		  //           = 117.9648 / 4 = 29.4912 MHz          
  		  // I2CBRG = [(Fcy/Fscl)-(Fcy/1,111,111)]-1
  		  //       = [(29.4912 MHz / 100KHz)-(29.4912MHz / 1,111,111)] - 1
          //       = [(294.912)-(26.542)]-1
          //       = 268.37 - 1
  		  //       = 267 = 10BH
		  267);												// I2C Baudrate(Approx. = 100 KHz)

  // Initial I2C Interrupt Control
  ConfigIntI2C(MI2C_INT_OFF &								// Disabe Master I2C Interrupt
               SI2C_INT_OFF &								// Disabe Slave I2C Interrupt
               MI2C_INT_PRI_7 &								// Set Priority Interrupt of Master = 7 
			   SI2C_INT_PRI_7 );							// Set Priority Interrupt of Slave = 7                 
}

/************************/
/* Read RTC:DS1307 Time */
/************************/
unsigned char ReadDS1307(unsigned char Read_Addr)			// Read RTC:DS1307 Time
{
  unsigned char Get_Time;									// Time Buffer

  StartI2C();												// Send Start Condition
  while(I2CCONbits.SEN);									// Wait Start Complete
    
  // Write DS1307 ID Code = 1101000+W
  MasterWriteI2C(0xD0);										// Write DS1307 ID Code,Write
  while(I2CSTATbits.TBF);									// Wait Write ID Code Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  

  // Write Address of RTC:ds1307 For Read
  MasterWriteI2C(Read_Addr);								// Write RTC Address 
  while(I2CSTATbits.TBF);									// Wait Write Address Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  
       
  // Restart For Read DS1307 Data
  RestartI2C();												// Send Re-Start Condition
  while(I2CCONbits.RSEN);									// Wait Re-Start Complete
     
  // Write DS1307 ID Code = 1101000+R 
  MasterWriteI2C(0xD1);										// Write DS1307 ID Code,Read
  while(I2CSTATbits.TBF);									// Wait Write ID Code Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  
    
  Get_Time = MasterReadI2C();								// Read Time From RTC
  while(I2CSTATbits.RBF);									// Wait Read Data Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear 
        
  StopI2C();												// Send Stop Condition
  while(I2CCONbits.PEN);									// Wait Stop Complete    

  return Get_Time;											// Return Time Result
}

/*************************/
/* Write RTC:DS1307 Time */
/*************************/
void WriteDS1307(unsigned char Write_Addr,					// Write RTC:DS1307 Time
                unsigned char Set_Time)						
{
  StartI2C();												// Send Start Condition
  while(I2CCONbits.SEN);									// Wait Start Complete

  // Write DS1307 ID Code = 1101000+W  
  MasterWriteI2C(0xD0);										// Write DS1307 ID Code (1101000+W)
  while(I2CSTATbits.TBF);									// Wait Write ID Code Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear     
         
  // Write RTC Address
  MasterWriteI2C(Write_Addr);								// Write RTC Address
  while(I2CSTATbits.TBF);									// Wait Write Address Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  

  // Write Time to RTC:DS1307   
  MasterWriteI2C(Set_Time);									// Write Time to RTC:DS1307
  while(I2CSTATbits.TBF);									// Wait Write Time Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  
  
  StopI2C();												// Send Stop Condition
  while(I2CCONbits.PEN);									// Wait Stop Complete
}
